import tkinter as tk
from tkinter import ttk
import random
import time
from threading import Thread

class SortingVisualizer:
    def __init__(self, root):
        self.root = root
        self.root.title("Визуализатор сортировки")
        self.root.geometry("950x700")
        
        # Массив для сортировки
        self.array = []
        self.sorting = False
        self.paused = False
        self.stop_sorting = False
        
        # Счетчики
        self.comparisons = 0
        self.swaps = 0
        
        # Настройки визуализации
        self.canvas_width = 850
        self.canvas_height = 400
        self.bar_width = 10
        self.min_value = 10
        self.max_value = 100
        
        self.setup_ui()
        self.generate_new_array()
    
    def setup_ui(self):
        # Основной фрейм
        main_frame = tk.Frame(self.root)
        main_frame.pack(fill=tk.BOTH, expand=True, padx=10, pady=10)
        
        # Фрейм управления
        control_frame = tk.Frame(main_frame)
        control_frame.pack(fill=tk.X, pady=5)
        
        # Левая часть управления - настройки генерации
        left_control = tk.Frame(control_frame)
        left_control.pack(side=tk.LEFT, fill=tk.X, expand=True)
        
        # Настройки размера массива
        size_frame = tk.Frame(left_control)
        size_frame.pack(fill=tk.X, pady=2)
        
        tk.Label(size_frame, text="Количество элементов:").pack(side=tk.LEFT)
        self.size_var = tk.IntVar(value=30)
        size_scale = tk.Scale(size_frame, from_=10, to=100, resolution=5,
                             orient=tk.HORIZONTAL, variable=self.size_var,
                             length=150, showvalue=True)
        size_scale.pack(side=tk.LEFT, padx=5)
        
        # Настройки диапазона значений
        range_frame = tk.Frame(left_control)
        range_frame.pack(fill=tk.X, pady=2)
        
        tk.Label(range_frame, text="Диапазон значений:").pack(side=tk.LEFT)
        tk.Label(range_frame, text="от").pack(side=tk.LEFT, padx=2)
        self.min_val_var = tk.IntVar(value=10)
        min_spin = tk.Spinbox(range_frame, from_=1, to=1000, textvariable=self.min_val_var,
                             width=5, command=self.validate_range)
        min_spin.pack(side=tk.LEFT, padx=2)
        
        tk.Label(range_frame, text="до").pack(side=tk.LEFT, padx=2)
        self.max_val_var = tk.IntVar(value=100)
        max_spin = tk.Spinbox(range_frame, from_=1, to=1000, textvariable=self.max_val_var,
                             width=5, command=self.validate_range)
        max_spin.pack(side=tk.LEFT, padx=2)
        
        # Правая часть управления - алгоритмы и управление
        right_control = tk.Frame(control_frame)
        right_control.pack(side=tk.RIGHT, fill=tk.Y)
        
        # Выбор алгоритма сортировки
        algo_frame = tk.Frame(right_control)
        algo_frame.pack(fill=tk.X, pady=2)
        
        tk.Label(algo_frame, text="Алгоритм:").pack(side=tk.LEFT)
        self.algorithm_var = tk.StringVar(value="Пузырьковая")
        algorithms = ["Пузырьковая", "Выбором", "Вставками", "Быстрая", "Слиянием"]
        
        algorithm_menu = ttk.Combobox(algo_frame, textvariable=self.algorithm_var,
                                     values=algorithms, state="readonly", width=12)
        algorithm_menu.pack(side=tk.LEFT, padx=5)
        
        # Скорость сортировки
        speed_frame = tk.Frame(right_control)
        speed_frame.pack(fill=tk.X, pady=2)
        
        tk.Label(speed_frame, text="Скорость:").pack(side=tk.LEFT)
        self.speed_var = tk.DoubleVar(value=0.1)
        speed_scale = tk.Scale(speed_frame, from_=0.01, to=1.0, resolution=0.01,
                              orient=tk.HORIZONTAL, variable=self.speed_var,
                              length=150, showvalue=True)
        speed_scale.pack(side=tk.LEFT, padx=5)
        
        # Фрейм кнопок управления
        button_frame = tk.Frame(main_frame)
        button_frame.pack(fill=tk.X, pady=10)
        
        # Кнопка генерации нового массива
        tk.Button(button_frame, text="Новый массив", 
                 command=self.generate_new_array, width=15).pack(side=tk.LEFT, padx=5)
        
        # Кнопка старта сортировки
        self.start_button = tk.Button(button_frame, text="Начать сортировку", 
                                     command=self.start_sorting, width=15, bg="lightgreen")
        self.start_button.pack(side=tk.LEFT, padx=5)
        
        # Кнопка паузы/продолжения
        self.pause_button = tk.Button(button_frame, text="Пауза", 
                                     command=self.toggle_pause, width=10, state=tk.DISABLED)
        self.pause_button.pack(side=tk.LEFT, padx=5)
        
        # Кнопка остановки
        self.stop_button = tk.Button(button_frame, text="Стоп", 
                                    command=self.stop_sorting_process, width=10, state=tk.DISABLED)
        self.stop_button.pack(side=tk.LEFT, padx=5)
        
        # Фрейм для счетчиков
        counters_frame = tk.Frame(main_frame)
        counters_frame.pack(fill=tk.X, pady=5)
        
        # Счетчик сравнений
        self.comparisons_label = tk.Label(counters_frame, text="Сравнений: 0", 
                                         font=("Arial", 12), fg="blue")
        self.comparisons_label.pack(side=tk.LEFT, padx=20)
        
        # Счетчик обменов
        self.swaps_label = tk.Label(counters_frame, text="Обменов: 0", 
                                   font=("Arial", 12), fg="red")
        self.swaps_label.pack(side=tk.LEFT, padx=20)
        
        # Счетчик времени
        self.time_label = tk.Label(counters_frame, text="Время: 0.00с", 
                                  font=("Arial", 12), fg="green")
        self.time_label.pack(side=tk.LEFT, padx=20)
        
        # Информационная метка
        self.info_label = tk.Label(main_frame, text="Сгенерирован новый массив", 
                                  font=("Arial", 12), bg="lightyellow")
        self.info_label.pack(fill=tk.X, pady=5)
        
        # Холст для отрисовки
        canvas_frame = tk.Frame(main_frame)
        canvas_frame.pack(fill=tk.BOTH, expand=True, pady=10)
        
        self.canvas = tk.Canvas(canvas_frame, width=self.canvas_width, 
                               height=self.canvas_height, bg="white")
        self.canvas.pack(fill=tk.BOTH, expand=True)
        
        # Фрейм для отображения массива
        array_frame = tk.Frame(main_frame)
        array_frame.pack(fill=tk.X)
        
        tk.Label(array_frame, text="Массив:").pack(side=tk.LEFT)
        self.array_label = tk.Label(array_frame, text="", font=("Courier", 10))
        self.array_label.pack(side=tk.LEFT, padx=5)
    
    def validate_range(self):
        """Проверка корректности диапазона значений"""
        if self.min_val_var.get() >= self.max_val_var.get():
            self.max_val_var.set(self.min_val_var.get() + 1)
    
    def reset_counters(self):
        """Сброс счетчиков"""
        self.comparisons = 0
        self.swaps = 0
        self.start_time = time.time()
        self.update_counters_display()
    
    def update_counters_display(self):
        """Обновление отображения счетчиков"""
        self.comparisons_label.config(text=f"Сравнений: {self.comparisons}")
        self.swaps_label.config(text=f"Обменов: {self.swaps}")
        
        # Обновление времени
        if hasattr(self, 'start_time') and self.sorting:
            elapsed = time.time() - self.start_time
            self.time_label.config(text=f"Время: {elapsed:.2f}с")
    
    def generate_new_array(self):
        if self.sorting:
            return
        
        size = self.size_var.get()
        min_val = self.min_val_var.get()
        max_val = self.max_val_var.get()
        
        # Корректировка диапазона если нужно
        if min_val >= max_val:
            max_val = min_val + 1
            self.max_val_var.set(max_val)
        
        self.array = [random.randint(min_val, max_val) for _ in range(size)]
        self.max_value = max_val
        self.reset_counters()
        self.update_display()
        self.info_label.config(text=f"Сгенерирован новый массив: {size} элементов, значения от {min_val} до {max_val}")
    
    def update_display(self, highlight_indices=None, compare_indices=None):
        self.canvas.delete("all")
        
        if highlight_indices is None:
            highlight_indices = []
        if compare_indices is None:
            compare_indices = []
        
        if not self.array:
            return
            
        # Вычисляем масштабирование
        bar_width = self.canvas_width / len(self.array)
        scale_factor = self.canvas_height / self.max_value
        
        # Рисуем столбцы
        for i, value in enumerate(self.array):
            x1 = i * bar_width
            y1 = self.canvas_height
            x2 = (i + 1) * bar_width
            y2 = self.canvas_height - value * scale_factor
            
            # Выбор цвета в зависимости от состояния элемента
            if i in highlight_indices:
                color = "red"  # Элементы для обмена
            elif i in compare_indices:
                color = "orange"  # Элементы для сравнения
            else:
                color = "lightblue"  # Обычные элементы
            
            self.canvas.create_rectangle(x1, y1, x2, y2, fill=color, outline="black")
            
            # Подписываем значения (только для достаточно широких столбцов)
            if bar_width > 25:
                self.canvas.create_text((x1 + x2) / 2, y2 - 15, text=str(value), font=("Arial", 8))
        
        # Обновляем текстовое представление массива
        array_text = "[" + ", ".join(map(str, self.array[:15])) + ("..." if len(self.array) > 15 else "") + "]"
        self.array_label.config(text=array_text)
        
        # Обновляем счетчики
        self.update_counters_display()
        
        self.root.update()
    
    def start_sorting(self):
        if self.sorting or not self.array:
            return
        
        self.sorting = True
        self.paused = False
        self.stop_sorting = False
        self.reset_counters()
        
        # Обновление состояния кнопок
        self.start_button.config(state=tk.DISABLED)
        self.pause_button.config(state=tk.NORMAL, text="Пауза", bg="lightyellow")
        self.stop_button.config(state=tk.NORMAL, bg="lightcoral")
        
        algorithm = self.algorithm_var.get()
        
        # Запускаем в отдельном потоке чтобы не блокировать GUI
        thread = Thread(target=self.run_sorting_algorithm, args=(algorithm,))
        thread.daemon = True
        thread.start()
    
    def toggle_pause(self):
        """Переключение паузы"""
        if not self.sorting:
            return
            
        self.paused = not self.paused
        if self.paused:
            self.pause_button.config(text="Продолжить", bg="lightgreen")
            self.info_label.config(text="Сортировка на паузе")
        else:
            self.pause_button.config(text="Пауза", bg="lightyellow")
            self.info_label.config(text="Сортировка продолжается...")
    
    def stop_sorting_process(self):
        """Остановка сортировки"""
        self.stop_sorting = True
        self.paused = False
        self.finish_sorting()
        self.info_label.config(text="Сортировка остановлена")
    
    def finish_sorting(self):
        """Завершение сортировки и сброс состояния"""
        self.sorting = False
        self.paused = False
        self.stop_sorting = False
        
        # Обновление состояния кнопок
        self.start_button.config(state=tk.NORMAL)
        self.pause_button.config(state=tk.DISABLED, text="Пауза", bg="SystemButtonFace")
        self.stop_button.config(state=tk.DISABLED, bg="SystemButtonFace")
    
    def wait_if_paused(self):
        """Ожидание если сортировка на паузе"""
        while self.paused and not self.stop_sorting:
            time.sleep(0.1)
            self.update_counters_display()
        return self.stop_sorting
    
    def run_sorting_algorithm(self, algorithm):
        self.info_label.config(text=f"Выполняется {algorithm} сортировка...")
        self.start_time = time.time()
        
        try:
            if algorithm == "Пузырьковая":
                self.bubble_sort()
            elif algorithm == "Выбором":
                self.selection_sort()
            elif algorithm == "Вставками":
                self.insertion_sort()
            elif algorithm == "Быстрая":
                self.quick_sort_wrapper()
            elif algorithm == "Слиянием":
                self.merge_sort_wrapper()
            
            if not self.stop_sorting:
                elapsed = time.time() - self.start_time
                self.info_label.config(
                    text=f"Сортировка завершена! Время: {elapsed:.2f}с, "
                         f"сравнений: {self.comparisons}, обменов: {self.swaps}"
                )
        
        except Exception as e:
            self.info_label.config(text=f"Ошибка при сортировке: {str(e)}")
        
        finally:
            self.finish_sorting()
    
    def bubble_sort(self):
        """Пузырьковая сортировка"""
        n = len(self.array)
        for i in range(n):
            for j in range(0, n - i - 1):
                if self.wait_if_paused():
                    return
                    
                # Сравнение элементов
                self.comparisons += 1
                self.update_display(compare_indices=[j, j + 1])
                time.sleep(self.speed_var.get())
                
                if self.array[j] > self.array[j + 1]:
                    # Обмен элементов
                    self.swaps += 1
                    self.array[j], self.array[j + 1] = self.array[j + 1], self.array[j]
                    self.update_display(highlight_indices=[j, j + 1])
                    time.sleep(self.speed_var.get())
    
    def selection_sort(self):
        """Сортировка выбором"""
        n = len(self.array)
        for i in range(n):
            if self.wait_if_paused():
                return
                
            min_idx = i
            for j in range(i + 1, n):
                if self.wait_if_paused():
                    return
                    
                # Сравнение элементов
                self.comparisons += 1
                self.update_display(compare_indices=[j, min_idx])
                time.sleep(self.speed_var.get())
                
                if self.array[j] < self.array[min_idx]:
                    min_idx = j
            
            if min_idx != i:
                # Обмен элементов
                self.swaps += 1
                self.array[i], self.array[min_idx] = self.array[min_idx], self.array[i]
                self.update_display(highlight_indices=[i, min_idx])
                time.sleep(self.speed_var.get())
    
    def insertion_sort(self):
        """Сортировка вставками"""
        for i in range(1, len(self.array)):
            if self.wait_if_paused():
                return
                
            key = self.array[i]
            j = i - 1
            
            # Сравнение при поиске позиции
            self.comparisons += 1
            self.update_display(compare_indices=[j, i])
            time.sleep(self.speed_var.get())
            
            while j >= 0 and key < self.array[j]:
                if self.wait_if_paused():
                    return
                    
                self.array[j + 1] = self.array[j]
                self.swaps += 1  # Перемещение элемента считается обменом
                j -= 1
                self.update_display(highlight_indices=[j + 1, i])
                time.sleep(self.speed_var.get())
                
                if j >= 0:
                    # Сравнение в цикле while
                    self.comparisons += 1
                    self.update_display(compare_indices=[j, i])
                    time.sleep(self.speed_var.get())
            
            self.array[j + 1] = key
            self.update_display(highlight_indices=[j + 1, i])
            time.sleep(self.speed_var.get())
    
    def quick_sort_wrapper(self):
        """Обертка для быстрой сортировки"""
        self.quick_sort(0, len(self.array) - 1)
    
    def quick_sort(self, low, high):
        """Быстрая сортировка"""
        if self.wait_if_paused():
            return
            
        if low < high:
            pi = self.partition(low, high)
            self.quick_sort(low, pi - 1)
            self.quick_sort(pi + 1, high)
    
    def partition(self, low, high):
        """Вспомогательная функция для быстрой сортировки"""
        pivot = self.array[high]
        i = low - 1
        
        for j in range(low, high):
            if self.wait_if_paused():
                return i + 1
                
            # Сравнение с опорным элементом
            self.comparisons += 1
            self.update_display(compare_indices=[j, high])
            time.sleep(self.speed_var.get())
            
            if self.array[j] <= pivot:
                i += 1
                if i != j:
                    self.swaps += 1
                    self.array[i], self.array[j] = self.array[j], self.array[i]
                    self.update_display(highlight_indices=[i, j])
                    time.sleep(self.speed_var.get())
        
        if i + 1 != high:
            self.swaps += 1
            self.array[i + 1], self.array[high] = self.array[high], self.array[i + 1]
            self.update_display(highlight_indices=[i + 1, high])
            time.sleep(self.speed_var.get())
        return i + 1
    
    def merge_sort_wrapper(self):
        """Обертка для сортировки слиянием"""
        self.merge_sort(0, len(self.array) - 1)
    
    def merge_sort(self, left, right):
        """Сортировка слиянием"""
        if self.wait_if_paused():
            return
            
        if left < right:
            mid = (left + right) // 2
            self.merge_sort(left, mid)
            self.merge_sort(mid + 1, right)
            self.merge(left, mid, right)
    
    def merge(self, left, mid, right):
        """Слияние для сортировки слиянием"""
        left_part = self.array[left:mid + 1]
        right_part = self.array[mid + 1:right + 1]
        
        i = j = 0
        k = left
        
        while i < len(left_part) and j < len(right_part):
            if self.wait_if_paused():
                return
                
            # Сравнение элементов из двух половин
            self.comparisons += 1
            self.update_display(compare_indices=[left + i, mid + 1 + j])
            time.sleep(self.speed_var.get())
            
            if left_part[i] <= right_part[j]:
                if self.array[k] != left_part[i]:
                    self.swaps += 1
                self.array[k] = left_part[i]
                i += 1
            else:
                if self.array[k] != right_part[j]:
                    self.swaps += 1
                self.array[k] = right_part[j]
                j += 1
            
            self.update_display(highlight_indices=[k])
            time.sleep(self.speed_var.get())
            k += 1
        
        while i < len(left_part):
            if self.wait_if_paused():
                return
                
            if self.array[k] != left_part[i]:
                self.swaps += 1
            self.array[k] = left_part[i]
            i += 1
            self.update_display(highlight_indices=[k])
            time.sleep(self.speed_var.get())
            k += 1
        
        while j < len(right_part):
            if self.wait_if_paused():
                return
                
            if self.array[k] != right_part[j]:
                self.swaps += 1
            self.array[k] = right_part[j]
            j += 1
            self.update_display(highlight_indices=[k])
            time.sleep(self.speed_var.get())
            k += 1

def main():
    root = tk.Tk()
    app = SortingVisualizer(root)
    root.mainloop()

if __name__ == "__main__":
    main()
